export default DICOMwebClient;
export type InstanceMetadata = import("../types/types").InstanceMetadata;
/**
 * A callback with the request instance and metadata information
 * of the currently request being executed that should necessarily
 * return the given request optionally modified.
 */
export type RequestHook = Function;
/**
 * @typedef { import("../types/types").InstanceMetadata } InstanceMetadata
 */
/**
 * A callback with the request instance and metadata information
 * of the currently request being executed that should necessarily
 * return the given request optionally modified.
 * @typedef {function} RequestHook
 * @param {XMLHttpRequest} request - The original XMLHttpRequest instance.
 * @param {object} metadata - The metadata used by the request.
 */
/**
 * Class for interacting with DICOMweb RESTful services.
 */
export class DICOMwebClient {
    static _parseQueryParameters(params?: {}): string;
    /**
     * Asserts that a given media type is valid.
     *
     * @params {String} mediaType media type
     */
    static _assertMediaTypeIsValid(mediaType: any): void;
    /**
     * Parses media type and extracts its type and subtype.
     *
     * @param {String} mediaType - HTTP media type (e.g. image/jpeg)
     * @private
     * @returns {String[]} Media type and subtype
     */
    private static _parseMediaType;
    /**
     * Builds an accept header field value for HTTP GET request messages.
     *
     * @param {Object[]} mediaTypes - Acceptable media types
     * @param {Object[]} supportedMediaTypes - Supported media types
     * @return {*}
     * @private
     */
    private static _buildAcceptHeaderFieldValue;
    /**
     * Builds an accept header field value for HTTP GET multipart request
     * messages.  Will throw an exception if no media types are found which are acceptable,
     * but will only log a verbose level message when types are specified which are
     * not acceptable.  This allows requesting several types with having to know
     * whether they are all acceptable or not.
     *
     * @param {Object[]} mediaTypes - Acceptable media types
     * @param {Object[]} supportedMediaTypes - Supported media types
     * @private
     */
    private static _buildMultipartAcceptHeaderFieldValue;
    /**
     * Builds a range header field value for HTTP GET request messages.
     *
     * @param {Array} byteRange - Start and end of byte range
     * @returns {String} Range header field value
     * @private
     */
    private static _buildRangeHeaderFieldValue;
    /**
     * Gets types that are shared among acceptable media types.
     *
     * @param {Object[]} mediaTypes - Acceptable media types and optionally the UIDs of the
     corresponding transfer syntaxes
     * @private
     * @returns {String[]} Types that are shared among acceptable media types
     */
    private static _getSharedMediaTypes;
    /**
     * Gets common base type of acceptable media types and asserts that only
     one type is specified. For example, ``("image/jpeg", "image/jp2")``
     will pass, but ``("image/jpeg", "video/mpeg2")`` will raise an
     exception.
     *
     * @param {Object[]} mediaTypes - Acceptable media types and optionally the UIDs of the
     corresponding transfer syntaxes
     * @private
     * @returns {String[]} Common media type, eg `image/` for the above example.
     */
    private static _getCommonMediaType;
    /**
     * @constructor
     * @param {Object} options
     * @param {String} options.url - URL of the DICOMweb RESTful Service endpoint
     * @param {String=} options.qidoURLPrefix - URL path prefix for QIDO-RS
     * @param {String=} options.wadoURLPrefix - URL path prefix for WADO-RS
     * @param {String=} options.stowURLPrefix - URL path prefix for STOW-RS
     * @param {String=} options.username - Username
     * @param {String=} options.password - Password
     * @param {Object=} options.headers - HTTP headers
     * @param {Array.<RequestHook>=} options.requestHooks - Request hooks.
     * @param {Object=} options.verbose - print to console request warnings and errors, default true
     * @param {Object=} options.debug - print to the console debug level information/status updates.
     * @param {boolean|String} options.singlepart - retrieve singlepart for the named types.
     * The available types are:  bulkdata, video, image.  true means all.
     */
    constructor(options: {
        url: string;
        qidoURLPrefix?: string | undefined;
        wadoURLPrefix?: string | undefined;
        stowURLPrefix?: string | undefined;
        username?: string | undefined;
        password?: string | undefined;
        headers?: any | undefined;
        requestHooks?: Array<RequestHook> | undefined;
        verbose?: any | undefined;
        debug?: any | undefined;
        singlepart: boolean | string;
    });
    baseURL: string;
    username: string;
    password: string;
    qidoURL: string;
    wadoURL: string;
    stowURL: string;
    singlepart: string;
    requestHooks: Function[];
    headers: any;
    errorInterceptor: any;
    verbose: boolean;
    /**
     * Allows setting the debug log information.
     * Note this is different from verbose in that verbose is whether to include warning/error information, defaulting to true
     *
     * @param {boolean} debugLevel
     * @param {function} debugLogFunction to call with the debug output arguments.
     */
    setDebug(debugLevel?: boolean, debugLogFunction?: Function): void;
    debugLevel: boolean;
    /**
     * Gets debug flag
     *
     * @returns true if debug logging is enabled
     */
    getDebug(): boolean;
    /**
     * Sets verbose flag.
     *
     * @param {Boolean} verbose
     */
    setVerbose(verbose: boolean): void;
    /**
     * Gets verbose flag.
     *
     * @return {Boolean} verbose
     */
    getVerbose(): boolean;
    /**
     * Performs an HTTP request.
     *
     * @param {String} url
     * @param {String} method
     * @param {Object} headers
     * @param {Object} options
     * @param {Array.<RequestHook>} options.requestHooks - Request hooks.
     * @param {XMLHttpRequest} [options.request] - if specified, the request to use, otherwise one will be created; useful for adding custom upload and abort listeners/objects
     * @return {*}
     * @private
     */
    private _httpRequest;
    /**
     * Performs an HTTP GET request.
     *
     * @param {String} url
     * @param {Object} headers
     * @param {Object} responseType
     * @param {Function} progressCallback
     * @return {*}
     * @private
     */
    private _httpGet;
    /**
     * Performs an HTTP GET request that accepts a message with application/json
     * media type.
     *
     * @param {String} url
     * @param {Object} params
     * @param {Function} progressCallback
     * @return {*}
     * @private
     */
    private _httpGetApplicationJson;
    /**
     * Performs an HTTP GET request that accepts a message with application/pdf
     * media type.
     *
     * @param {String} url
     * @param {Object} params
     * @param {Function} progressCallback
     * @return {*}
     * @private
     */
    private _httpGetApplicationPdf;
    /**
     * Performs an HTTP GET request that accepts a message with an image
     media type.
     *
     * @param {String} url
     * @param {Object[]} mediaTypes
     * @param {Object} params
     * @param {Function} progressCallback
     * @return {*}
     * @private
     */
    private _httpGetImage;
    /**
     * Performs an HTTP GET request that accepts a message with a text
     media type.
     *
     * @param {String} url
     * @param {Object[]} mediaTypes
     * @param {Object} params
     * @param {Function} progressCallback
     * @return {*}
     * @private
     */
    private _httpGetText;
    /**
     * Performs an HTTP GET request that accepts a message with a video
     media type.
     *
     * @param {String} url
     * @param {Object[]} mediaTypes
     * @param {Object} params
     * @param {Function} progressCallback
     * @return {*}
     * @private
     */
    private _httpGetVideo;
    /**
     * Performs an HTTP GET request that accepts a multipart message with an image media type.
     *
     * @param {String} url - Unique resource locator
     * @param {Object[]} mediaTypes - Acceptable media types and optionally the UIDs of the
     corresponding transfer syntaxes
     * @param {Array} byteRange - Start and end of byte range
     * @param {Object} params - Additional HTTP GET query parameters
     * @param {Boolean} rendered - Whether resource should be requested using rendered media types
     * @param {Function} progressCallback
     * @private
     * @returns {Promise<Array>} Content of HTTP message body parts
     */
    private _httpGetMultipartImage;
    /**
     * Performs an HTTP GET request that accepts a multipart message with a video media type.
     *
     * @param {String} url - Unique resource locator
     * @param {Object[]} mediaTypes - Acceptable media types and optionally the UIDs of the
     corresponding transfer syntaxes
     * @param {Array} byteRange - Start and end of byte range
     * @param {Object} params - Additional HTTP GET query parameters
     * @param {Boolean} rendered - Whether resource should be requested using rendered media types
     * @param {Function} progressCallback
     * @private
     * @returns {Promise<Array>} Content of HTTP message body parts
     */
    private _httpGetMultipartVideo;
    /**
     * Performs an HTTP GET request that accepts a multipart message
     * with a application/dicom media type.
     *
     * @param {String} url - Unique resource locator
     * @param {Object[]} mediaTypes - Acceptable media types and optionally the UIDs of the
     corresponding transfer syntaxes
     * @param {Object} params - Additional HTTP GET query parameters
     * @param {Function} progressCallback
     * @private
     * @returns {Promise<Array>} Content of HTTP message body parts
     */
    private _httpGetMultipartApplicationDicom;
    /**
     * Performs an HTTP GET request that accepts a multipart message
     * with a application/octet-stream, OR any of the equivalencies for that (eg
     * application/pdf etc)
     *
     * @param {String} url - Unique resource locator
     * @param {Object[]} mediaTypes - Acceptable media types and optionally the UIDs of the
     corresponding transfer syntaxes
     * @param {Array} byteRange start and end of byte range
     * @param {Object} params - Additional HTTP GET query parameters
     * @param {Function} progressCallback
     * @private
     * @returns {Promise<Array>} Content of HTTP message body parts
     */
    private _httpGetMultipartApplicationOctetStream;
    /**
     * Performs an HTTP POST request.
     *
     * @param {String} url - Unique resource locator
     * @param {Object} headers - HTTP header fields
     * @param {Array} data - Data that should be stored
     * @param {Function} progressCallback
     * @param {Function} progressCallback
     * @param {XMLHttpRequest} request - if specified, the request to use, otherwise one will be created; useful for adding custom upload and abort listeners/objects
     * @private
     * @returns {Promise} Response
     */
    private _httpPost;
    /**
     * Performs an HTTP POST request with content-type application/dicom+json.
     *
     * @param {String} url - Unique resource locator
     * @param {Object} headers - HTTP header fields
     * @param {Array} data - Data that should be stored
     * @param {Function} progressCallback
     * @private
     * @returns {Promise} Response
     */
    private _httpPostApplicationJson;
    /**
     * Searches for DICOM studies.
     *
     * @param {Object} options
     * @param {Object} [options.queryParams] - HTTP query parameters
     * @return {Object[]} Study representations (http://dicom.nema.org/medical/dicom/current/output/chtml/part18/sect_6.7.html#table_6.7.1-2)
     */
    searchForStudies(options?: {
        queryParams?: any;
    }): any[];
    /**
     * Retrieves metadata for a DICOM study.
     *
     * @param {Object} options
     * @param {String} options.studyInstanceUID - Study Instance UID
     * @returns {Promise<InstanceMetadata[]>} Metadata elements in DICOM JSON format for each instance
                        belonging to the study
     */
    retrieveStudyMetadata(options: {
        studyInstanceUID: string;
    }): Promise<InstanceMetadata[]>;
    /**
     * Searches for DICOM series.
     *
     * @param {Object} options
     * @param {Object} [options.studyInstanceUID] - Study Instance UID
     * @param {Object} [options.queryParams] - HTTP query parameters
     * @returns {Object[]} Series representations (http://dicom.nema.org/medical/dicom/current/output/chtml/part18/sect_6.7.html#table_6.7.1-2a)
     */
    searchForSeries(options?: {
        studyInstanceUID?: any;
        queryParams?: any;
    }): any[];
    /**
     * Retrieves metadata for a DICOM series.
     *
     * @param {Object} options
     * @param {String} options.studyInstanceUID - Study Instance UID
     * @param {String} options.seriesInstanceUID - Series Instance UID
     * @returns {Promise<InstanceMetadata[]>} Metadata elements in DICOM JSON format for each instance
                        belonging to the series
     */
    retrieveSeriesMetadata(options: {
        studyInstanceUID: string;
        seriesInstanceUID: string;
    }): Promise<InstanceMetadata[]>;
    /**
     * Searches for DICOM Instances.
     *
     * @param {Object} options
     * @param {String} [options.studyInstanceUID] - Study Instance UID
     * @param {String} [options.seriesInstanceUID] - Series Instance UID
     * @param {Object} [options.queryParams] - HTTP query parameters
     * @returns {Object[]} Instance representations (http://dicom.nema.org/medical/dicom/current/output/chtml/part18/sect_6.7.html#table_6.7.1-2b)
     */
    searchForInstances(options?: {
        studyInstanceUID?: string;
        seriesInstanceUID?: string;
        queryParams?: any;
    }): any[];
    /** Returns a WADO-URI URL for an instance
     *
     * @param {Object} options
     * @param {String} options.studyInstanceUID - Study Instance UID
     * @param {String} options.seriesInstanceUID - Series Instance UID
     * @param {String} options.sopInstanceUID - SOP Instance UID
     * @returns {String} WADO-URI URL
     */
    buildInstanceWadoURIUrl(options: {
        studyInstanceUID: string;
        seriesInstanceUID: string;
        sopInstanceUID: string;
    }): string;
    /**
     * Retrieves metadata for a DICOM Instance.
     *
     * @param {Object} options object
     * @param {String} options.studyInstanceUID - Study Instance UID
     * @param {String} options.seriesInstanceUID - Series Instance UID
     * @param {String} options.sopInstanceUID - SOP Instance UID
     * @returns {Promise<InstanceMetadata>} metadata elements in DICOM JSON format
     */
    retrieveInstanceMetadata(options: {
        studyInstanceUID: string;
        seriesInstanceUID: string;
        sopInstanceUID: string;
    }): Promise<InstanceMetadata>;
    /**
     * Retrieves frames for a DICOM Instance.
     * @param {Object} options options object
     * @param {String} options.studyInstanceUID - Study Instance UID
     * @param {String} options.seriesInstanceUID - Series Instance UID
     * @param {String} options.sopInstanceUID - SOP Instance UID
     * @param {String} options.frameNumbers - One-based indices of Frame Items
     * @returns {Array} frame items as byte arrays of the pixel data element
     */
    retrieveInstanceFrames(options: {
        studyInstanceUID: string;
        seriesInstanceUID: string;
        sopInstanceUID: string;
        frameNumbers: string;
    }): any[];
    /**
    * Element in mediaTypes parameter
    * @typedef {Object} MediaType
    * @param {String} [MediaType.mediaType] - ie 'image/jpeg', 'image/png'...
    */
    /**
     * Retrieves an individual, server-side rendered DICOM Instance.
     *
     * @param {Object} options
     * @param {String} options.studyInstanceUID - Study Instance UID
     * @param {String} options.seriesInstanceUID - Series Instance UID
     * @param {String} options.sopInstanceUID - SOP Instance UID
     * @param {MediaType[]} [options.mediaTypes] - Acceptable HTTP media types
     * @param {Object} [options.queryParams] - HTTP query parameters
     * @returns {Promise<ArrayBuffer>} Rendered DICOM Instance
     */
    retrieveInstanceRendered(options: {
        studyInstanceUID: string;
        seriesInstanceUID: string;
        sopInstanceUID: string;
        mediaTypes?: any[];
        queryParams?: any;
    }): Promise<ArrayBuffer>;
    /**
     * Retrieves a thumbnail of an DICOM Instance.
     *
     * @param {Object} options
     * @param {String} options.studyInstanceUID - Study Instance UID
     * @param {String} options.seriesInstanceUID - Series Instance UID
     * @param {String} options.sopInstanceUID - SOP Instance UID
     * @param {MediaType[]} [options.mediaTypes] - Acceptable HTTP media types
     * @param {Object} [options.queryParams] - HTTP query parameters
     * @returns {ArrayBuffer} Thumbnail
     */
    retrieveInstanceThumbnail(options: {
        studyInstanceUID: string;
        seriesInstanceUID: string;
        sopInstanceUID: string;
        mediaTypes?: any[];
        queryParams?: any;
    }): ArrayBuffer;
    /**
     * Retrieves rendered frames for a DICOM Instance.
     *
     * @param {Object} options
     * @param {String} options.studyInstanceUID - Study Instance UID
     * @param {String} options.seriesInstanceUID - Series Instance UID
     * @param {String} options.sopInstanceUID - SOP Instance UID
     * @param {String} options.frameNumbers - One-based indices of Frame Items
     * @param {MediaType[]} [options.mediaTypes] - Acceptable HTTP media types
     * @param {Object} [options.queryParams] - HTTP query parameters
     * @returns {ArrayBuffer[]} Rendered Frame Items as byte arrays
     */
    retrieveInstanceFramesRendered(options: {
        studyInstanceUID: string;
        seriesInstanceUID: string;
        sopInstanceUID: string;
        frameNumbers: string;
        mediaTypes?: any[];
        queryParams?: any;
    }): ArrayBuffer[];
    /**
     * Retrieves thumbnail of frames for a DICOM Instance.
     *
     * @param {Object} options
     * @param {String} options.studyInstanceUID - Study Instance UID
     * @param {String} options.seriesInstanceUID - Series Instance UID
     * @param {String} options.sopInstanceUID - SOP Instance UID
     * @param {String} options.frameNumbers - One-based indices of Frame Items
     * @param {Object} [options.queryParams] - HTTP query parameters
     * @returns {ArrayBuffer[]} Rendered Frame Items as byte arrays
     */
    retrieveInstanceFramesThumbnail(options: {
        studyInstanceUID: string;
        seriesInstanceUID: string;
        sopInstanceUID: string;
        frameNumbers: string;
        queryParams?: any;
    }): ArrayBuffer[];
    /**
     * Retrieves a DICOM Instance.
     *
     * @param {Object} options
     * @param {String} options.studyInstanceUID - Study Instance UID
     * @param {String} options.seriesInstanceUID - Series Instance UID
     * @param {String} options.sopInstanceUID - SOP Instance UID
     * @returns {Promise<ArrayBuffer>} DICOM Part 10 file as Arraybuffer
     */
    retrieveInstance(options: {
        studyInstanceUID: string;
        seriesInstanceUID: string;
        sopInstanceUID: string;
    }): Promise<ArrayBuffer>;
    /**
     * Retrieves all DICOM Instances of a Series.
     *
     * @param {Object} options
     * @param {String} options.studyInstanceUID - Study Instance UID
     * @param {String} options.seriesInstanceUID - Series Instance UID
     * @param {Function} options.progressCallback
     * @returns {Promise<ArrayBuffer[]>} DICOM Instances
     */
    retrieveSeries(options: {
        studyInstanceUID: string;
        seriesInstanceUID: string;
        progressCallback: Function;
    }): Promise<ArrayBuffer[]>;
    /**
     * Retrieves all DICOM Instances of a Study.
     *
     * @param {Object} options
     * @param {String} options.studyInstanceUID - Study Instance UID
     * @returns {ArrayBuffer[]} DICOM Instances
     */
    retrieveStudy(options: {
        studyInstanceUID: string;
    }): ArrayBuffer[];
    /**
     * Retrieves and parses BulkData from a BulkDataURI location.
     * Decodes the multipart encoded data and returns the resulting data
     * as an ArrayBuffer.
     *
     * See http://dicom.nema.org/medical/dicom/current/output/chtml/part18/sect_6.5.5.html
     *
     * @param {Object} options
     * @param {string} options.BulkDataURI to retrieve
     * @param {Array}  options.mediaTypes to use to fetch the URI
     * @param {string} options.byteRange to request a sub-range (only valid on single part)
     * @returns {Promise<Array>} Bulkdata parts
     */
    retrieveBulkData(options: {
        BulkDataURI: string;
        mediaTypes: any[];
        byteRange: string;
    }): Promise<any[]>;
    /**
     * Stores DICOM Instances.
     *
     * @param {Object} options
     * @param {ArrayBuffer[]} options.datasets - DICOM Instances in PS3.10 format
     * @param {String} [options.studyInstanceUID] - Study Instance UID
     * @param {XMLHttpRequest} [options.request] - if specified, the request to use, otherwise one will be created; useful for adding custom upload and abort listeners/objects
     * @returns {Promise} Response message
     */
    storeInstances(options: {
        datasets: ArrayBuffer[];
        studyInstanceUID?: string;
        request?: XMLHttpRequest;
    }): Promise<any>;
}
//# sourceMappingURL=api.d.ts.map