import i18n from "i18next";
import { id } from "./id";
import initToolGroups from "./initToolGroups";
import toolbarButtons from "./toolbarButtons";

// Allow this mode by excluding non-imaging modalities such as SR, SEG
// Also, SM is not a simple imaging modalities, so exclude it.
const NON_IMAGE_MODALITIES = ["ECG", "SEG", "RTSTRUCT", "RTPLAN", "PR"];

const ohif = {
  layout: "@ohif/extension-default.layoutTemplateModule.viewerLayout",
  sopClassHandler: "@ohif/extension-default.sopClassHandlerModule.stack",
  thumbnailList: "@ohif/extension-default.panelModule.seriesList",
  wsiSopClassHandler:
    "@ohif/extension-cornerstone.sopClassHandlerModule.DicomMicroscopySopClassHandler",
};

const cornerstone = {
  measurements: "@ohif/extension-cornerstone.panelModule.panelMeasurement",
  segmentation: "@ohif/extension-cornerstone.panelModule.panelSegmentation",
};

const tracked = {
  measurements:
    "@ohif/extension-measurement-tracking.panelModule.trackedMeasurements",
  thumbnailList: "@ohif/extension-measurement-tracking.panelModule.seriesList",
  viewport:
    "@ohif/extension-measurement-tracking.viewportModule.cornerstone-tracked",
};

const dicomsr = {
  sopClassHandler:
    "@ohif/extension-cornerstone-dicom-sr.sopClassHandlerModule.dicom-sr",
  sopClassHandler3D:
    "@ohif/extension-cornerstone-dicom-sr.sopClassHandlerModule.dicom-sr-3d",
  viewport: "@ohif/extension-cornerstone-dicom-sr.viewportModule.dicom-sr",
};

const dicomvideo = {
  sopClassHandler:
    "@ohif/extension-dicom-video.sopClassHandlerModule.dicom-video",
  viewport: "@ohif/extension-dicom-video.viewportModule.dicom-video",
};

const dicompdf = {
  sopClassHandler: "@ohif/extension-dicom-pdf.sopClassHandlerModule.dicom-pdf",
  viewport: "@ohif/extension-dicom-pdf.viewportModule.dicom-pdf",
};

const dicomSeg = {
  sopClassHandler:
    "@ohif/extension-cornerstone-dicom-seg.sopClassHandlerModule.dicom-seg",
  viewport: "@ohif/extension-cornerstone-dicom-seg.viewportModule.dicom-seg",
};

const dicomPmap = {
  sopClassHandler:
    "@ohif/extension-cornerstone-dicom-pmap.sopClassHandlerModule.dicom-pmap",
  viewport: "@ohif/extension-cornerstone-dicom-pmap.viewportModule.dicom-pmap",
};

const dicomRT = {
  viewport: "@ohif/extension-cornerstone-dicom-rt.viewportModule.dicom-rt",
  sopClassHandler:
    "@ohif/extension-cornerstone-dicom-rt.sopClassHandlerModule.dicom-rt",
};

const extensionDependencies = {
  // Can derive the versions at least process.env.from npm_package_version
  "@ohif/extension-default": "^3.0.0",
  "@ohif/extension-cornerstone": "^3.0.0",
  "@ohif/extension-measurement-tracking": "^3.0.0",
  "@ohif/extension-cornerstone-dicom-sr": "^3.0.0",
  "@ohif/extension-cornerstone-dicom-seg": "^3.0.0",
  "@ohif/extension-cornerstone-dicom-pmap": "^3.0.0",
  "@ohif/extension-cornerstone-dicom-rt": "^3.0.0",
  "@ohif/extension-dicom-pdf": "^3.0.1",
  "@ohif/extension-dicom-video": "^3.0.1",
};

function modeFactory({ modeConfiguration }) {
  let _activatePanelTriggersSubscriptions = [];
  return {
    // TODO: We're using this as a route segment
    // We should not be.
    id,
    routeName: "viewer",
    displayName: i18n.t("Modes:Basic Viewer"),
    /**
     * Lifecycle hooks
     */
    onModeEnter: function ({
      servicesManager,
      extensionManager,
      commandsManager,
    }: withAppTypes) {
      const {
        measurementService,
        toolbarService,
        toolGroupService,
        customizationService,
      } = servicesManager.services;

      measurementService.clearMeasurements();

      // Init Default and SR ToolGroups
      initToolGroups(extensionManager, toolGroupService, commandsManager);

      toolbarService.register(toolbarButtons);
      toolbarService.updateSection(toolbarService.sections.primary, [
        "StackScroll",
        "Zoom",
        "Pan",
        "Layout",
        "Length",
        "Bidirectional",
        "ArrowAnnotate",
        "EllipticalROI",
        "RectangleROI",
        "CircleROI",
        "PlanarFreehandROI",
        "SplineROI",
        "LivewireContour",
        // 'MeasurementTools',

        "TrackballRotate",
        "WindowLevel",
        "Capture",
        "Crosshairs",
        // 'MoreTools',
        "Reset",
        "rotate-right",
        "flipHorizontal",
        "ImageSliceSync",
        "ReferenceLines",
        "ImageOverlayViewer",
        "invert",
        "Probe",
        "Cine",
        "Angle",
        "CobbAngle",
        "Magnify",
        "CalibrationLine",
        "TagBrowser",
        "AdvancedMagnify",
        "UltrasoundDirectionalTool",
        "WindowLevelRegion",
        "SegmentLabelTool",
      ]);

      toolbarService.updateSection(
        toolbarService.sections.viewportActionMenu.topLeft,
        ["orientationMenu", "dataOverlayMenu"]
      );

      toolbarService.updateSection(
        toolbarService.sections.viewportActionMenu.bottomMiddle,
        ["AdvancedRenderingControls"]
      );

      toolbarService.updateSection("AdvancedRenderingControls", [
        "windowLevelMenuEmbedded",
        "voiManualControlMenu",
        "Colorbar",
        "opacityMenu",
        "thresholdMenu",
      ]);

      toolbarService.updateSection(
        toolbarService.sections.viewportActionMenu.topRight,
        ["modalityLoadBadge", "trackingStatus", "navigationComponent"]
      );

      toolbarService.updateSection(
        toolbarService.sections.viewportActionMenu.bottomLeft,
        ["windowLevelMenu"]
      );

      toolbarService.updateSection("MeasurementTools", [
        "Length",
        "Bidirectional",
        "ArrowAnnotate",
        "EllipticalROI",
        "RectangleROI",
        "CircleROI",
        "PlanarFreehandROI",
        "SplineROI",
        "LivewireContour",
      ]);

      toolbarService.updateSection("MoreTools", [
        "Reset",
        "rotate-right",
        "flipHorizontal",
        "ImageSliceSync",
        "ReferenceLines",
        "ImageOverlayViewer",
        "StackScroll",
        "invert",
        "Probe",
        "Cine",
        "Angle",
        "CobbAngle",
        "Magnify",
        "CalibrationLine",
        "TagBrowser",
        "AdvancedMagnify",
        "UltrasoundDirectionalTool",
        "WindowLevelRegion",
        "SegmentLabelTool",
      ]);

      customizationService.setCustomizations({
        "panelSegmentation.disableEditing": {
          $set: true,
        },
      });

      // // ActivatePanel event trigger for when a segmentation or measurement is added.
      // // Do not force activation so as to respect the state the user may have left the UI in.
      // _activatePanelTriggersSubscriptions = [
      //   ...panelService.addActivatePanelTriggers(
      //     cornerstone.segmentation,
      //     [
      //       {
      //         sourcePubSubService: segmentationService,
      //         sourceEvents: [segmentationService.EVENTS.SEGMENTATION_ADDED],
      //       },
      //     ],
      //     true
      //   ),
      //   ...panelService.addActivatePanelTriggers(
      //     tracked.measurements,
      //     [
      //       {
      //         sourcePubSubService: measurementService,
      //         sourceEvents: [
      //           measurementService.EVENTS.MEASUREMENT_ADDED,
      //           measurementService.EVENTS.RAW_MEASUREMENT_ADDED,
      //         ],
      //       },
      //     ],
      //     true
      //   ),
      //   true,
      // ];
    },
    onModeExit: ({ servicesManager }: withAppTypes) => {
      const {
        toolGroupService,
        syncGroupService,
        segmentationService,
        cornerstoneViewportService,
        uiDialogService,
        uiModalService,
      } = servicesManager.services;

      _activatePanelTriggersSubscriptions.forEach((sub) => sub.unsubscribe());
      _activatePanelTriggersSubscriptions = [];

      uiDialogService.hideAll();
      uiModalService.hide();
      toolGroupService.destroy();
      syncGroupService.destroy();
      segmentationService.destroy();
      cornerstoneViewportService.destroy();
    },
    validationTags: {
      study: [],
      series: [],
    },

    isValidMode: function ({ modalities }) {
      const modalities_list = modalities.split("\\");

      // Exclude non-image modalities
      return {
        valid: !!modalities_list.filter(
          (modality) => NON_IMAGE_MODALITIES.indexOf(modality) === -1
        ).length,
        description:
          "The mode does not support studies that ONLY include the following modalities: SM, ECG, SEG, RTSTRUCT",
      };
    },
    routes: [
      {
        path: "longitudinal",
        /*init: ({ servicesManager, extensionManager }) => {
          //defaultViewerRouteInit
        },*/
        layoutTemplate: () => {
          return {
            id: ohif.layout,
            props: {
              leftPanels: [tracked.thumbnailList],
              leftPanelResizable: true,
              rightPanels: [cornerstone.segmentation, tracked.measurements],
              rightPanelClosed: true,
              rightPanelResizable: true,
              leftPanelClosed: window.matchMedia("(pointer: coarse)").matches,
              viewports: [
                {
                  namespace: tracked.viewport,
                  displaySetsToDisplay: [
                    ohif.sopClassHandler,
                    dicomvideo.sopClassHandler,
                    ohif.wsiSopClassHandler,
                  ],
                },
                {
                  namespace: dicomsr.viewport,
                  displaySetsToDisplay: [
                    dicomsr.sopClassHandler,
                    dicomsr.sopClassHandler3D,
                  ],
                },
                {
                  namespace: dicompdf.viewport,
                  displaySetsToDisplay: [dicompdf.sopClassHandler],
                },
                {
                  namespace: dicomSeg.viewport,
                  displaySetsToDisplay: [dicomSeg.sopClassHandler],
                },
                {
                  namespace: dicomPmap.viewport,
                  displaySetsToDisplay: [dicomPmap.sopClassHandler],
                },
                {
                  namespace: dicomRT.viewport,
                  displaySetsToDisplay: [dicomRT.sopClassHandler],
                },
              ],
            },
          };
        },
      },
    ],
    extensions: extensionDependencies,
    // Default protocol gets self-registered by default in the init
    hangingProtocol: "default",
    // Order is important in sop class handlers when two handlers both use
    // the same sop class under different situations.  In that case, the more
    // general handler needs to come last.  For this case, the dicomvideo must
    // come first to remove video transfer syntax before ohif uses images
    sopClassHandlers: [
      dicomvideo.sopClassHandler,
      dicomSeg.sopClassHandler,
      dicomPmap.sopClassHandler,
      ohif.sopClassHandler,
      ohif.wsiSopClassHandler,
      dicompdf.sopClassHandler,
      dicomsr.sopClassHandler3D,
      dicomsr.sopClassHandler,
      dicomRT.sopClassHandler,
    ],
    ...modeConfiguration,
  };
}

const mode = {
  id,
  modeFactory,
  extensionDependencies,
};

export default mode;
export { initToolGroups, toolbarButtons };
