import React, { useEffect, useMemo, useState } from "react";
import { useToolbar } from "@ohif/core";

interface ToolbarProps {
  buttonSection?: string;
  viewportId?: string;
  location?: number;
}

type ToolUsage = {
  id: string;
  count: number;
  lastUsed: number;
};

export function Toolbar({
  buttonSection = "primary",
  viewportId,
  location,
}: ToolbarProps) {
  const {
    toolbarButtons,
    onInteraction,
    isItemOpen,
    isItemLocked,
    openItem,
    closeItem,
    toggleLock,
  } = useToolbar({
    buttonSection,
  });

  const [usageData, setUsageData] = useState<Record<string, ToolUsage>>({});

  useEffect(() => {
    const stored = localStorage.getItem("toolUsage");
    if (stored) {
      setUsageData(JSON.parse(stored));
    }
  }, []);

  useEffect(() => {
    localStorage.setItem("toolUsage", JSON.stringify(usageData));
  }, [usageData]);

  // Create a sorted array of tool IDs instead of sorting components directly
  // const sortedIds = useMemo(() => {
  //   if (!toolbarButtons.length) {
  //     return null;
  //   }

  //   return toolbarButtons
  //     .map((tool, index) => ({
  //       id: tool.id,
  //       index, // original index
  //       count: usageData[tool.id]?.count ?? 0,
  //     }))
  //     .sort((a, b) => {
  //       if (a.count === 0 && b.count === 0) {
  //         // keep original order if both unused
  //         return a.index - b.index;
  //       }
  //       return b.count - a.count; // higher count first
  //     })
  //     .map((item) => item.id);
  // }, [toolbarButtons, usageData]);

  const sortedIds = useMemo(() => {
    if (!toolbarButtons.length) {
      return null;
    }
    // keep original order — no sorting
    return toolbarButtons.map((tool) => tool.id);
  }, [toolbarButtons]);

  if (!toolbarButtons.length) {
    return null;
  }

  return (
    <>
      {sortedIds.map((id) => {
        const toolDef = toolbarButtons.find((t) => t.id === id);
        if (!toolDef) return null;

        const { Component, componentProps } = toolDef;

        const enhancedProps = {
          ...componentProps,
          isOpen: isItemOpen(id, viewportId),
          isLocked: isItemLocked(id, viewportId),
          onOpen: () => openItem(id, viewportId),
          onClose: () => closeItem(id, viewportId),
          onToggleLock: () => toggleLock(id, viewportId),
          viewportId,
        };

        return (
          <div key={id}>
            <Component
              id={id}
              location={location}
              onInteraction={(args) => {
                setUsageData((prev) => {
                  const prevTool = prev[id] || { id, count: 0, lastUsed: 0 };
                  return {
                    ...prev,
                    [id]: {
                      ...prevTool,
                      count: prevTool.count + 1,
                      lastUsed: Date.now(),
                    },
                  };
                });

                onInteraction({
                  ...args,
                  itemId: id,
                  viewportId,
                });
              }}
              {...enhancedProps}
            />
          </div>
        );
      })}
    </>
  );
}
